// screen.c

// (C) by Satyria.de -> see https://satyria.de/arm under Raspberry Pi4
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

#include "config.h"
#include "screen.h"
#include "mailbox.h"
#include "base.h"
#include "util.h"
#include "types.h"
//#include <stdint.h>
//#include "led.h"
#include <ms8x8font.h>


u32 pScreen[36] ALIGN(16) = {
    34*4, //34 Einträge * 4 Bytes
    0,
    Set_physical_display,
    0x8,0x8,
    SCREEN_X,
    SCREEN_Y,
    
    Set_virtual_buffer,
    0x8,0x8,
    SCREEN_X,
    SCREEN_Y,

    Set_depth,
    0x4,0x4,
    BITS_PER_PIXEL,
    
    Set_virtual_offset,
    0x8,0x8,
    0x0,0x0,
    
    Set_palette,
    0x10,0x10,
    0x0,
    0x2,
    0x00000000,
    0xffffffff,
    
    Allocate_buffer,
    0x8,0x8,
    0x0,
    0x0,

    0x00000000
};

u32 graphicsAddress = 0;
u32 DrawColor = 0xFFFFFFFF;
u32 FrontColor = 0xFFFFFFFF;
u32 BackColor = 0;

#define ADDRESS_MASK 0x3FFFFFFF

void Init_Screen(void) 
{

    uintptr pScreenAddress = (uintptr)pScreen;
    while (TRUE) {
        BcmMailBox_Write(BCM_MAILBOX_PROP_OUT,(u32)pScreenAddress);
        
        if (pScreen[31] != 0)
        {
            break;
        }
    }
    
    // Passt die Adresse an und speichert sie
    graphicsAddress = pScreen[31] & ADDRESS_MASK;
	 FrontColor = 0xFFFFFFFF;
	 BackColor = 0;
}

void DrawPixel(u32 x, u32 y)
{
    if ((x < SCREEN_X) && (y < SCREEN_Y))
    {
        write32(DrawColor,graphicsAddress+(((SCREEN_X*y)+x)*4));
    }
}

u32 GetPixel(u32 x, u32 y)
{
    if ((x < SCREEN_X) && (y < SCREEN_Y))
    {
        return read32(graphicsAddress+(((SCREEN_X*y)+x)*4));
    }
    else
    {
        return 0;
    }
}

void SetDrawColor(u8 r, u8 g, u8 b)
{
    DrawColor = (0xFF << 24) | (r << 16) | (g << 8) | b;
}

void SetFrontColor(u8 r, u8 g, u8 b)
{
    FrontColor = (0xFF << 24) | (r << 16) | (g << 8) | b;
}

void SetBackColor(u8 r, u8 g, u8 b)
{
    BackColor = (0xFF << 24) | (r << 16) | (g << 8) | b;
}

void DrawChar(char c, u32 x0, u32 y0)
{
    unsigned char *sym = font[(unsigned char)c];
    
    for (u32 i = 0; i < FONT_HEIGHT; i++)
    {
        for (u32 j = 0; j < FONT_WIDTH; j++)
        {
            u32 check = (sym[i] >> (7 - j)) & 1;
            
            if (check == 1)
            {
                DrawColor = FrontColor;
            }
            else
            {
                DrawColor = BackColor;
            }
            
            DrawPixel(x0 + j, y0 + i);
        }
    }
}

#define CHAR_WIDTH 8
#define CHAR_HEIGHT 10
#define NUM_COLS (SCREEN_X / CHAR_WIDTH)
#define NUM_ROWS (SCREEN_Y / CHAR_HEIGHT)

u32 cursor_x = 0;
u32 cursor_y = 0;

void ClearScreen()
{
    for (u32 y = 0; y < SCREEN_Y; y++)
    {
        for (u32 x = 0; x < SCREEN_X; x++)
        {
            DrawPixel(x, y);
        }
    }
    cursor_x = 0;
    cursor_y = 0;
}

void MoveCursor(u32 x, u32 y)
{
    cursor_x = x;
    cursor_y = y;
}

void ScrollScreen()
{
    for (u32 y = 0; y < SCREEN_Y - CHAR_HEIGHT; y++)
    {
        for (u32 x = 0; x < SCREEN_X; x++)
        {
            DrawColor = GetPixel(x, y + CHAR_HEIGHT);
            DrawPixel(x, y);
        }
    }
    // Clear the last line
    DrawColor = BackColor;
    for (u32 y = SCREEN_Y - CHAR_HEIGHT; y < SCREEN_Y; y++)
    {
        for (u32 x = 0; x < SCREEN_X; x++)
        {
            DrawPixel(x, y);
        }
    }
}

void DrawCharAtCursor(char c)
{
    if (c == '\n') //(New Line) Zeilenumbruch
    {
        cursor_x = 0;
        cursor_y++;
    }
    else if (c == '\r') //(Carriage Return): Setzt den Cursor an den Anfang der aktuellen Zeile.
    {
        cursor_x = 0;
    }
    else if (c == '\t') //(Horizontal Tab) Bewegt den Cursor auf die nächste Tab-Position.
    {
        cursor_x += 4;
    }
    else if (c == '\b') //(Backspace) Bewegt den Cursor um ein Zeichen nach links.
    {
        if (cursor_x > 0)
        {
            cursor_x--;
        }
    }
    else if (c == '\f') //(Form Feed) Neue Seite (kann z.B. verwendet werden, um den Bildschirm zu löschen).
    {
        ClearScreen();
    }
    else if (c == '\033')
    { // ESC-Zeichen erkannt, sollte Abgefangen werden.
        
    }
    else
    {
        DrawChar(c, cursor_x * CHAR_WIDTH, cursor_y * CHAR_HEIGHT);
        cursor_x++;
        if (cursor_x >= NUM_COLS)
        {
            cursor_x = 0;
            cursor_y++;
        }
    }
    if (cursor_y >= NUM_ROWS) 
    {
        ScrollScreen();
        cursor_y = NUM_ROWS - 1;
    }
}

void DrawString(const char* str) {
    while (*str)
    {
        DrawCharAtCursor(*str++);
    }
}
